---------------------------------------------------------------------------------
--                             PAT-Tester-Script                               --
--                    using script-base version 2023-05-08                     --
--                    requires channel-firmware V113 or higher                 --
---------------------------------------------------------------------------------



--#############################################################################--
--##                                                                         ##--
--##            P A R T  1 :     S T A T E   M A C H I N E                   ##--
--##                                                                         ##--
--#############################################################################--

-------------------- Generate Step Configuration --------------------------------
-- This function contains the defintion of the script steps.
-- It is called before every step transition, to re-calculate any math expressions.
-- The user can overwrite and call this function at _any_ time during measurement: 
--   * Changes to limits will become effective immediately.
--   * All other changes will become effective after the next step transition.
--   * It is possible to force a step transition by calling SwitchTo().
---------------------------------------------------------------------------------
function Generate_StepConfiguration()

-- weight section --
-- m1 = ...
-- m2 = ...
-- end of weight section --

-- configuration section --



-- end of configuration section --
    
end

-------------------- Initialization ---------------------------------------------
-- This function is called only once to configure the steps,
-- initialize all global variables, and prepare the output files.
---------------------------------------------------------------------------------
function Init_StateMachine()

    -- script runtime data --
    t            = 0      -- * statemachine time since script init / Seconds
    step_n       = 0      --   statemachine step number
    step_t       = 0      -- * statemachine step time, for use as PAT-Composer limit only
    cycle_n      = 0      -- * statemachine cycle number
    cycle_t      = 0      -- * statemachine cycle time, for use as PAT-Composer limit only
    step_tstart  = 0      --   statemachine time when the step was started / Seconds
    cycle_tstart = 0      --   statemachine time when the cycle was started / Seconds
    started      = false  --   statemachine has not yet started
    finished     = false  --   statemachine is not finished
    processing   = false  --   script is not processing data
    paused       = false  --   script is not paused
    stopped      = false  --   script is not aborted
    progress     = 0      --   script progress in percent
    
    -- step runtime data --
    -- Explanation of the config structures:
    -- "stepconf":  The "stepconf" structure contains the step config provided by the user.
    --              It can be overwritten by the user at any time during script execution.
    --              All math expressions are repeatedly evaluated before every step transition.
    --              The script itself shall not modify "stepconf" - all changes would be lost.
    -- "stepstate": The "stepstate" structure is private to the script. 
    --              It acts as a state memory, i.e. for loop counters.

    -- initialize new step states --
    stepstate = {}  -- create step state for non constant values 
    
end

function AddMissingTables(n)

    -- stepconf --
    if stepconf[n].record == nil then stepconf[n].record = {V12 = 0.0001, V1R = 0.001, V2R = 0.001, i = 0.00001, t = 999999} end
    if stepconf[n].limits == nil then stepconf[n].limits = {} end
        
    -- stepstate --
    if stepstate[n]        == nil then stepstate[n] = {} end
    if stepstate[n].limits == nil then stepstate[n].limits = {[0] = {loops = 0}} end
    
end

-------------------- Start Measurement ------------------------------------------
-- Does some self-test and more Init and then starts the state machine.
---------------------------------------------------------------------------------
function StartMeasurement()

    -- check channel firmware version --
    if ec.GetHardwareInfo then board_type, board_rev, firmw_vers = ec.GetHardwareInfo() end
    if firmw_vers == nil or firmw_vers < 113 then ec.print(SEV_ERR, "Please update channel firmware to >=113") end
  
    -- add missing step tables--
    for n,step in pairs(stepconf) do
        AddMissingTables(n)
    end
    
    DebugMemoryAllocation()

    -- set safety limits --
    ec.SetSafetyLimits( safetylimits.V12_min,
                        safetylimits.V12_max,
                        safetylimits.i,
                        safetylimits.delay_count )

    -- reset time to 0 seconds --
    ec.ResetSeconds()
    
    -- start measurement by switching to the first step --
    step_n = -1
    SwitchNext()
    
    InitSamplingStepState()
end

function DebugMemoryAllocation()

    -- free some memory --
    Init_SampleProcessing = nil
    Init_StateMachine = nil
    
    -- do a dry run through all steps --
    --ec.print(SEV_INFO, "Starting a dry run of the script")
    --dryrun = true
    --for n,step in pairs(stepconf) do
    --    SwitchTo(n)  -- includes GC calls
    --end
    --dryrun = false
    
    -- debug print size of globals table
    local i = 0
    local j = 0
    for k,v in ipairs(_G) do i = i + 1 end
    for k,v in pairs(_G) do j = j + 1 end
    ec.print(SEV_INFO, "Debug size(_G): "..i.."+"..(j-i))
    
    -- check if enough memory is left --
    -- stops execution here if memory is insufficient--
    for i = 1, 20 do _G["dummy"..i] = "dummy" end  -- allocates 20 string variables (~1KB) to resize globals table
    long_string1 = string.rep(" ", 10000)          -- allocates one memory block of 10K
    long_string2 = string.rep(" ", 10000)          -- allocates one memory block of 10K
    long_string1 = nil                             -- free again
    long_string2 = nil                             -- free again
    for i = 1, 20 do _G["dummy"..i] = nil end      -- free again
    
end

------------------- Check Limits / invoke the state machine ---------------------
--  Checks the step limits and switches to the specified step if a limit is exceeded.
---------------------------------------------------------------------------------
function CheckLimits()

    t       = ec.GetSeconds()      -- * statemachine time since script init / Seconds
    step_t  = t - step_tstart      -- * statemachine step time, for use as PAT-Composer limit only
    cycle_t = t - cycle_tstart     -- * statemachine cycle time, for use as PAT-Composer limit only
    
    -- return if sample is from old step --
    if (sample_t > step_tstart) then  -- "if (sample_step_n == step_n)" would not work if step loops to itself

        -- call limit functions in stepconf[n].limits --
        if stepconf[sample_step_n].limits then
            for m, Limitfunction in pairs(stepconf[sample_step_n].limits) do
                -- shorter pointer name for stepstate limits table
                local l = stepstate[sample_step_n].limits
                -- create empty limit tables if not yet existing
                if not l[m] then l[m] = {} end
                -- initialize limit loop counter if not yet existing
                l[m].loops = l[m].loops or 0
                -- call limit functions
                Limitfunction(l[m])  -- checks limit and eventually calls SwitchNext() or SwitchTo()
            end
        end

    end
end

----------------------------- Increment Loops -----------------------------------
--  Called from a function that's defined in stepconf config block,
--  this function counts the loops of a step limit.
---------------------------------------------------------------------------------
function IncLoops(limit)
    limit.loops = limit.loops + 1
end

------------------------- Switch To Next Step -----------------------------------
--  Switches to the next step.
--  This function is usually called when a limit has been exceeded.
---------------------------------------------------------------------------------
function SwitchNext()
    for next_step_n = step_n + 1, 1000 do   -- look for next step
        if stepconf[next_step_n] then
            SwitchTo(next_step_n)
            return
        end
    end
    -- if no next step exists
    SwitchTo(step_n + 1)
end

------------------------- Switch To Step n --------------------------------------
--  Switches to the step number specified in the argument.
--  This function is usually called when a limit has been exceeded.
---------------------------------------------------------------------------------
function SwitchTo(...)

    if type(...) ~= 'table' then
        n_newstep = assert(...)
    else
        local t = ...  -- table pointer
        n_newstep = assert(t[1])
    end
    
    sample_rec = true  -- keep the sample that triggered the limit (even though it may not be the last sample of the step)

    -- execute user code here --
    if stepconf[step_n] and stepconf[step_n].doAtStepEnd then 
       stepconf[step_n].doAtStepEnd(step_n) 
    end
    if stepconf[n_newstep] and stepconf[n_newstep].doPreInit then 
       stepconf[n_newstep].doPreInit(n_newstep) 
    end
    
    -- assign "last-step" values needed for Generate_StepConfiguration()
    V12_last_step = V12
    V1R_last_step = V1R
    V2R_last_step = V2R
    i12_last_step = i12
    i1R_last_step = i1R
    i2R_last_step = i2R
    
    -- rebuild stepconf, re-evaluate expressions --
    stepconf = nil
    collectgarbage('collect')
    Generate_StepConfiguration()
    
    if not stepconf[n_newstep] then 
        -- if new step does not exist, go into OFF mode and stop --
        ec.print(SEV_INFO, "Step does not exist. Switching OFF.")
        stepconf[n_newstep] = {}  -- create new step "OFF"
        stepconf[n_newstep].step_type = "OFF"
    end
    
    -- copy parameter list into new step configuration --
    if type(...) == 'table' then
        local t = ...  -- table pointer
        t[1] = nil     -- delete step number
        for parameter, value in pairs(t) do
            ec.print(SEV_INFO, "... overwriting parameter: "..parameter.." = "..value)
            stepconf[n_newstep][parameter] = value
        end
    end
    
    AddMissingTables(n_newstep)

    -- configure new hardware state (as pending changes only, until ec.SwitchNow is called ) -------------

    -- assign new step config (or use config from last step, or use default values) --
    step_type        = stepconf[n_newstep].step_type   or "Rest"
    step_con         = stepconf[n_newstep].con         or "1vs2"
    step_range       = stepconf[n_newstep].range       or "auto"
    step_bw          = stepconf[n_newstep].bw          or "slow"
    step_diffcap_dV          = stepconf[n_newstep].diffcap_dV          or 0.001
    step_diffcap_startdelayR = stepconf[n_newstep].diffcap_startdelayR or 5
    step_diffcap_startdelay  = stepconf[n_newstep].diffcap_startdelay  or 5
    step_slope        = stepconf[n_newstep].slope        or 0
    step_pg           = "-" -- not yet determined
    step_record_V12   = stepconf[n_newstep].record.V12   or 0.0001
    step_record_V1R   = stepconf[n_newstep].record.V1R   or 0.001
    step_record_V2R   = stepconf[n_newstep].record.V2R   or 0.001
    step_record_i     = stepconf[n_newstep].record.i     or 0.00001
    step_record_t     = stepconf[n_newstep].record.t     or 999999
    step_rec_delay_ms = stepconf[n_newstep].rec_delay_ms or 0
    
    -- create defaults for v and i (last measured values) --
    local v_last_step
    local i_last_step
    if (step_con == "1vsR") then
        v_last_step = V1R_last_step
        i_last_step = i12_last_step
    elseif (step_con == "2vsR") then
        v_last_step = V2R_last_step
        i_last_step = i12_last_step
    elseif (step_con == "Rvs1") then
        v_last_step = V1R_last_step
        i_last_step = i1R_last_step
    elseif (step_con == "Rvs2") then
        v_last_step = V2R_last_step
        i_last_step = i2R_last_step
    else
        v_last_step = V12_last_step
        i_last_step = i12_last_step
    end
    step_v_ctrl = stepconf[n_newstep].v or (v_last_step or nil)
    step_i_ctrl = stepconf[n_newstep].i or (i_last_step or 0)
    
    -- configure step type and connection --
    cd_guess = nil
    if step_type == "OFF" then
        step_pg = "-"
        step_con = "OFF"
        
    elseif step_type == "OCV" or step_type == "Rest" then
        step_pg = "-"
        step_con = "OCV"
        
    elseif step_type == "CC" then
        step_pg = "g"
        ec.PG_SetConstCurrent(step_i_ctrl)  -- current in Amps, not Milliamps
        if step_i_ctrl > 0 then cd_guess = "c" end
        if step_i_ctrl < 0 then cd_guess = "d" end
        
    elseif step_type == "CV" then 
        step_pg = "p"
        ec.PG_SetConstVoltage(step_v_ctrl)
        
    elseif step_type == "GEIS" then 
        step_pg = "g"
        ec.PG_SetConstCurrent(step_i_ctrl)
        if step_i_ctrl > 0 then cd_guess = "c" end
        if step_i_ctrl < 0 then cd_guess = "d" end
        
    elseif step_type == "PEIS" then 
        step_pg = "p"
        ec.PG_SetConstVoltage(step_v_ctrl)
        
    elseif step_type == "VS" then 
        step_pg = "p"
        ec.PG_SetConstVoltage(step_v_ctrl)
        if step_slope > 0 then cd_guess = "c" end
        if step_slope < 0 then cd_guess = "d" end
        
    else
        ec.print(SEV_WARN, "step_type '"..step_type.."' not recognized. Switching to 'OCV' instead.")
        step_pg = "-"
        step_con = "OCV" -- open circuit voltage
    end
    
    -- execute user code here --
    if stepconf[n_newstep].doPostInit then stepconf[n_newstep].doPostInit(n_newstep) end
    
    if not dryrun then 
    
        -- firmware calls --
        ec.PG_SetCurrentRange(step_range)
        ec.PG_SetBandwidth(step_bw)
        ec.SetCon(step_con)
        ec.ADC_RecDelta(step_record_V12, step_record_V1R, step_record_V2R, step_record_i, step_record_t)
        ec.ADC_RecStepDelay(step_rec_delay_ms)
        
        -- Apply the new hardware configuration now (switch matrix and PStat/GStat into new state) --
		ec.print(SEV_INFO, string.format("SwitchTo(%1.0f): %s (triggered at t=%1.3fs, switching at t=%1.3fs)", n_newstep, step_type, sample_t, ec.GetSeconds()))
        sample_step_tlast = ec.SwitchNow()   -- Firmware returns timestamp of last acquired A/D-sample before switching.
                                             -- At the time of switching, this sample resides in the A/D-sample-queue.
        -- remember step start time --
        step_tstart = ec.GetSeconds()

        step_n = n_newstep
        
        ------ cycle autodetection ------
        cds_user = stepconf[n_newstep].cycle_start or "auto" -- cycle starts with charge "c" or discharge "d"
        cd_user  = stepconf[n_newstep].cd          or "auto" -- cycle is charge "c" or discharge "d"
        -- remember settings of previous step --
        cds_prev_step = step_cds or "auto"
        cd_prev_step  = step_cd  or "auto"
        -- at this point, only allowed values are "c" and "d" and nil ("auto" equals nil) --
        if cds_user ~= "c" and cds_user ~= "d" then cds_user = nil end
        if cd_user  ~= "c" and cd_user  ~= "d" then cd_user  = nil end
        if step_cds ~= "c" and step_cds ~= "d" then step_cds = nil end
        if step_cd  ~= "c" and step_cd  ~= "d" then step_cd  = nil end
        -- select detection source, depending on priority
        step_cds = cds_user  -- prio 1: user cds value, if existing
                or step_cds  -- prio 2: previous step value, if existing (remain unchanged)
                or cd_user   -- prio 3: user cd value, if existing
                or cd_guess  -- prio 4: guessed cd value, if existing (based on CC set current or CV set slope)
                or "auto"    -- prio 5: default value
        step_cd =  cd_user   -- prio 1: user value, if existing
                or cd_guess  -- prio 2: guessed value, if existing (based on CC set current or CV set slope)
                or step_cd   -- prio 3: previous step value, if existing (remain unchanged)
                or "auto"    -- prio 4: default value
        if step_cds ~= cds_prev_step then
            ec.print(SEV_INFO, "Cycles start with '"..step_cds.."' (c=charge, d=discharge)")
        end
        -- new step will start a new cycle? --
        if((step_cds == "c" and cycle_n == 0 and step_cd == "c")                       -- start of first c-d  cycle
        or (step_cds == "d" and cycle_n == 0 and step_cd == "d")                       -- start of first d-c cycle
        or (step_cds == "c" and cd_prev_step == "d" and step_cd ~= cd_prev_step)       -- start of successive c-d cycle
        or (step_cds == "d" and cd_prev_step == "c" and step_cd ~= cd_prev_step)) then -- start of successive d-c cycle
            -- update statemachine cycle state
            cycle_n = cycle_n + 1
            cycle_tstart = step_tstart
            ec.print(SEV_INFO, "Cycle "..cycle_n.." start")
        end
        
        -- output script progress --
        ec.SetScriptProgress(cycle_n, step_n, progress)
        
        -- if step has special activities, start them here --
        if step_type == "VS" then 
            ec.WaveGenRamp(step_v_ctrl, step_slope)
        elseif step_type == "PEIS" or step_type == "GEIS" then
            StartEIS()
        elseif step_type == "OFF" and #(stepconf[n_newstep].limits) == 0 then 
            -- if OFF step has no limits, then statemachine is finished --
            ec.print(SEV_INFO, "Script finished")
            finished = true  
            -- state machine has finished, but processing continues
            -- until all remaining samples are processed and the last cycle is completed
        end
    end
    
    -- debug print channelboard heap memory allocation --
    collectgarbage('collect')
    local heap_allocation = collectgarbage('count')
    if heap_allocation > 200 then
        ec.print(SEV_INFO, string.format("%1.0f KB of RAM used", heap_allocation))
    end
end

----------------------------- Init EIS ------------------------------------------
-- Initialize the PEIS/GEIS Step and start the spectrum scan
---------------------------------------------------------------------------------
function StartEIS()

    ec.EIS_Spectrum(stepconf[step_n].eis_amplitude, 
                    stepconf[step_n].eis_fstart                          or 100000,
                    stepconf[step_n].eis_fend                            or 1 ,
                    stepconf[step_n].eis_fcount_per_ordermagnitude       or 3 , 
                    stepconf[step_n].eis_ordermagnitude                  or 10, 
                    stepconf[step_n].eis_averages                        or 1 ,
                    stepconf[step_n].eis_precycles                       or 1 ,
                    stepconf[step_n].eis_record_waveform_up_to_frequency or 1 ,
                    stepconf[step_n].eis_apply_drift_correction          or 0 ,
                    stepconf[step_n].geis_limit_V12_amplitude            or 0 ,
                    stepconf[step_n].geis_limit_V1R_amplitude            or 0 ,
                    stepconf[step_n].geis_limit_V2R_amplitude            or 0 ,
                    stepconf[step_n].peis_limit_i_amplitude              or 0 ,
                    stepconf[step_n].eis_sweep_back                      or 0 ,
                    stepconf[step_n].eis_measure_Z_AUX                   or 0 )
                    
    if step_bw ~= "fast" then
        ec.print(SEV_WARN, "PGStat bandwidth should be set to 'fast' in EIS steps.")
    end
end

--------------------------- Add Progress ----------------------------------------
-- This subroutine is called from a step limit function
-- and outputs a message containing the script status in percent.
---------------------------------------------------------------------------------
function AddProgress(val)
    progress = progress + val
end

-------------------- Pause and Resume -------------------------------------------
--  Switch cell connection to OCV or resume from there.
---------------------------------------------------------------------------------
function Pause()       
    if not paused and not stopped and not finished then
        ec.SetCon("OCV")
        ec.SwitchNow()
        paused = true
        ec.SetScriptExecutionState("Paused")
    end
end

function Resume()
    if paused and not stopped and not finished then
        ec.SetCon(stepconf[step_n].con)
        ec.SwitchNow()
        paused = false
        ec.SetScriptExecutionState("Resumed")
    end
end




--#############################################################################--
--##                                                                         ##--
--##       P A R T  2 :   A / D   S A M P L E   P R O C E S S I N G          ##--
--##                                                                         ##--
--#############################################################################--

---------------------------------------------------------------------------------
function Init_SampleProcessing()

    -------- GLOBAL VARIABLES --------
    
    -- A star * comment means that this variable is selectable in PAT-Composer.
    
    -- SAMPLE DATA --
    -- A/D-converter measured data (last sample data):
    V12              = 0     -- * sample voltage             / Volt
    V1R              = 0     -- * sample voltage             / Volt
    V2R              = 0     -- * sample voltage             / Volt
    i                = 0     --   sample current             / Ampere
    Vaux             = 0     -- * sample auxiliary voltage   / Volt
    sample_over      = 0     --   sample overrange warning (string " ", "i", "v" or "iv")
    sample_i_range   = 0     --   current range (1="100mA", 2="10mA", 3="1mA", 4="0.1mA")
    
    -- more data belonging to last A/D sample:
    -- notice: 
    --   All A/D samples are buffered - their step and cycle number may differ from the statemachine state.
    --   "Sample time" is the true timepoint of A/D-sampling, not the time of the sample processing.
    
    -- time data:
    sample_t         = 0     -- * sample time, counted in seconds since script init
    sample_t_prev    = 0     --   sample time of previous sample (for calculation of dt)
    sample_t_prev_display = 0 --  sample time of previous displayed sample (live plot update rate)
    sample_t_prev_record  = 0 --  sample time of previous recorded sample (saved to Output.txt) (becomes obsolete in 2023)
    sensor_sample_t_prev_record = 0 --  sample time of previous recorded sample (saved to Output-Sensordata.txt)
    sample_t_prev_diffcap = 0 --  sample time of previous differential-capacity calculation (to print 1sps maximum)
    sample_step_tstart  = 0  --   sample time of the first sample of the step
    sample_step_tlast   = 0  --   sample time of the last sample of the step
    sample_cycle_tstart = 0  --   sample time of the first sample of the cycle
    sample_step_t    = 0     -- * sample time in step
    sample_cycle_t   = 0     -- * sample time in cycle
    -- step config:
    sample_step_n    = 0     --   sample step  number
    sample_cycle_n   = 0     -- * sample cycle number
    sample_step_type = ""    --   step type can be: "OFF", "OCV", "CC", "CV", "VS", "PEIS", "GEIS", ... 
    sample_step_pg   = ""    --   potentiostat "p" or galvanostat "g", somewhat redundant with 'step_type'
    sample_cds       = ""    --   cycle starts with charge "c" or discharge "d"
    sample_cd        = ""    --   step is charge "c" or discharge "d"
    sample_step_con  = ""    --   electrode connection / configuration, i.e. "1vs2"
    sample_step_dV   = nil   --   differential capacity: constant delta V as in "dQ/dV"
    -- reduction algorithm:
    sample_rec       = false --   flag: record this sample (true) or do not record this sample (false)
    sample_queue     = 0     --   for debugging: Buffer fill level of A/D sample buffer
    record_maximum_dt= 3600  --   save at least one sample every N seconds (to Output.txt) (becomes obsolete in 2023, better use recording criterium "t", GitLab#141)
    display_maximum_dt= 1    --   show at least one new sample every N seconds in live plot (live update rate)
    record_sensor_dt = 1     --   save at least one sample of sensor data every N seconds (to Output-Sensordata.txt)
    -- calculated data:
    i12              = 0     -- * current between electrodes 1 and 2
    i1R              = 0     -- * current between electrodes 1 and R
    i2R              = 0     -- * current between electrodes 2 and R
    P                = 0     -- * power (in any electrode configuration)
    -- total charge and energy:
    Q12              = 0     --   total charge flown between electrodes 1 and 2
    Q1R              = 0     --   total charge flown between electrodes 1 and R 
    Q2R              = 0     --   total charge flown between electrodes 2 and R 
    Q1               = 0     -- * total charge on electrode 1 
    Q2               = 0     -- * total charge on electrode 2 
    QR               = 0     -- * total charge on electrode R 
    W12              = 0     -- * total energy
    -- cycle charge and energy:
    Qc               = 0     -- * cycle charged charge,    always positive, any configuration
    Wc               = 0     -- * cycle charged energy,    always positive, any configuration
    Qd               = 0     -- * cycle discharged charge, always positive, any configuration
    Wd               = 0     -- * cycle discharged energy, always positive, any configuration
    Qcd              = 0     -- * cycle charge, reset only between charge and discharge, any configuration
    Qcd_abs          = 0     -- * cycle Qcd absolute value, always positive, often called "capacity"
    EffQ             = 0     -- * Qd / Qc   
    EffW             = 0     -- * Wd / Wc   
    -- differential capacity:
    last_V_C12       = 0     --   last voltage on which dQ/|dV| has been calculated and saved
    last_V_C1R       = 0     --   last voltage on which dQ/dV has been calculated and saved
    last_V_C2R       = 0     --   last voltage on which dQ/dV has been calculated and saved
    last_V_C1        = 0     --   last voltage on which dQ/dV has been calculated and saved
    last_V_C2        = 0     --   last voltage on which dQ/dV has been calculated and saved
    last_V_CR        = 0     --   last voltage on which dQ/dV has been calculated and saved
    last_Q_C12       = 0     --   last charge on which dQ/dV has been calculated and saved
    last_Q_C1R       = 0     --   last charge on which dQ/dV has been calculated and saved
    last_Q_C2R       = 0     --   last charge on which dQ/dV has been calculated and saved
    last_Q_C1        = 0     --   last charge on which dQ/dV has been calculated and saved
    last_Q_C2        = 0     --   last charge on which dQ/dV has been calculated and saved
    last_Q_CR        = 0     --   last charge on which dQ/dV has been calculated and saved
    dQ_C12           = 0     -- * change in charge
    dQ_C1R           = 0     -- * change in charge
    dQ_C2R           = 0     -- * change in charge
    dQ_C1            = 0     -- * change in charge
    dQ_C2            = 0     -- * change in charge
    dQ_CR            = 0     -- * change in charge
    C12              = 0     -- * differential capacity dQ12/dV12 (with current i12)
    C1R              = 0     -- * differential capacity dQ1R/dV1R (with current i1R)
    C2R              = 0     -- * differential capacity dQ2R/dV2R (with current i2R)
    C1               = 0     -- * differential capacity dQ1/dV1R (with current i12) or dQ1/dV12 (with current i1R)
    C2               = 0     -- * differential capacity dQ2/dV2R (with current i12) or dQ2/dV12 (with current i2R)
    CR               = 0     -- * differential capacity dQR/dV2R (with current i1R) or dQR/dV1R (with current i2R)
    -- last voltages and currents from previous step:
    V12_last_step    = 0
    V1R_last_step    = 0
    V2R_last_step    = 0
    i12_last_step    = 0
    i1R_last_step    = 0
    i2R_last_step    = 0
    -- lowpass filter --
    V12_avg          = 0     --   voltage average over i.e. one second
    V1R_avg          = 0     --   voltage average over i.e. one second
    V2R_avg          = 0     --   voltage average over i.e. one second
    avg_counter      = 0     --   average filter sample counter
    -- voltage slope |dV/dt| --
    dV12_dt_abs      = 0     -- * |dV12/dt|
    dV1R_dt_abs      = 0     -- * |dV1R/dt|
    dV2R_dt_abs      = 0     -- * |dV2R/dt|
    delta_t_min      = 1     --   minimum dt time interval used to calculate slope dV/dt, in seconds
    delta_t_max      = 99999 --   maximum dt time interval used to calculate slope dV/dt, in seconds
    -- sensor data --
    T_set            = 0     --   temperature setpoint of chamber
    T_chamber        = 0     -- * temperature measured in chamber
    T_cell           = 0     -- * temperature measured in PAT-Cell
    T_board          = 0     --   temperature of channelboard
    T_ambient        = 0     --   temperature of chamber ambient air
    Pressure         = 0     -- * gas pressure [bar] in PAT-Cell-Press
    Dilation         = 0     -- * cell stack dilation in [m] (SI-unit) in ECD-4
    Force            = 0     -- * cell stack force [N]
    T                = 0     -- * temperature [degC], obsolete, for backwards compatibility only
    press            = 0     -- * gas pressure [bar], obsolete, for backwards compatibility only
    -- EIS data --
    EIS_freq         = 0     --   frequency
    EIS_t            = 0     --   time
    EIS_reserved     = 0
    EIS_V12_dc       = 0     --   average voltages
    EIS_V1R_dc       = 0
    EIS_V2R_dc       = 0
    EIS_i12_dc       = 0     --   average currents
    EIS_i1R_dc       = 0
    EIS_i2R_dc       = 0
    EIS_i_range      = 0
    EIS_Z12_re       = 0     --   impedance real part      Re(Z)
    EIS_Z12_im       = 0     --   impedance imaginary part Im(Z)
    EIS_Z12_mag      = 0     --   impedance magnitude      |Z|
    EIS_Z12_phi      = 0     --   impedance phase angle    phi(Z)
    EIS_dV12         = 0     --   amplitude of voltage at given frequency
    EIS_di12         = 0     --   amplitude of current at given frequency
    EIS_Z12_SFDR     = 0     --   Spurious Free Dynamic Range, a measure for impedance measurement quality
    EIS_Z1_re        = 0
    EIS_Z1_im        = 0
    EIS_Z1_mag       = 0
    EIS_Z1_phi       = 0
    EIS_dV1R         = 0
    EIS_Z1_SFDR      = 0
    EIS_Z2_re        = 0
    EIS_Z2_im        = 0
    EIS_Z2_mag       = 0
    EIS_Z2_phi       = 0
    EIS_dV2R         = 0
    EIS_Z2_SFDR      = 0
    EIS_Z1R_re       = 0
    EIS_Z1R_im       = 0
    EIS_Z1R_mag      = 0
    EIS_Z1R_phi      = 0
    EIS_di1R         = 0
    EIS_Z1R_SFDR     = 0
    EIS_Z2R_re       = 0
    EIS_Z2R_im       = 0
    EIS_Z2R_mag      = 0
    EIS_Z2R_phi      = 0
    EIS_di2R         = 0
    EIS_Z2R_SFDR     = 0
    EIS_Cs           = 0     --   capacitance calculated using an R+C (series) equivalent circuit
    EIS_Cp           = 0     --   capacitance calculated using an R|C (parallel) equivalent circuit
    EIS_precycles    = 0     --   count of sine-cycles generated before the start of each EIS single-frequency measurement
    EIS_averages     = 0     --   count of sine-cycles avaraged during each EIS single-frequency-measurement
    EIS_DAC          = 0     --   for debug only
    EIS_A            = 0     --   for debug only
    EIS_n            = 0     --   for debug only
    EIS_m            = 0     --   for debug only
    EIS_f_err        = 0     --   for debug only
    EIS_dt           = 0     --   for debug only

    -- some constants --
    SEV_ERR     = 0          --   info message severity: E r r o r  (fatal)
    SEV_WARN    = 1          --   info message severity: W a r n i n g
    SEV_INFO    = 2          --   info message severity: I n f o
    SEV_ERROR   = 0          --   obsolete, do not use
    SEV_WARNING = 1          --   obsolete, do not use

    -- input-data argument table placeholder --
    arg     = string.rep(" ", 500)   -- NewSample() arguments
    imp_arg = string.rep(" ", 2000)  -- NewEISData() arguments

    -- output file columns --
    tim_data = {}  -- time based data columns
    sen_data = {}  -- time based data columns of additional sensors
    C12_data = {}  -- differential capacity data columns
    C1R_data = {}  -- differential capacity data columns
    C2R_data = {}  -- differential capacity data columns
    C1_data  = {}  -- differential capacity data columns
    C2_data  = {}  -- differential capacity data columns
    CR_data  = {}  -- differential capacity data columns
    cyc_data = {}  -- per cycle data columns
    imp_data = {}  -- impedance data columns
    

    -- print file headers --
    local header = true
    PrintOutput(header)
    PrintOutputSensordata(header)
    PrintOutputCycles(header)
    PrintOutputImpedance(header)
    PrintOutputC12(header)
    PrintOutputC1R(header)
    PrintOutputC2R(header)
    PrintOutputC1(header)
    PrintOutputC2(header)    
    PrintOutputCR(header)
    
end


------------------ New Sample (function called by firmware) ---------------------
--  * Sample processing starts here - this is the entry point of sample processing.
--  * The channel firmware calls this routine 1042 times per second to pass measurement
--    data from the A/D-converter-sample-buffer to this script. 
--  * The flag "sample_rec" is a boolean variable that flags some of the many sample points
--    that are worth recording. The flag is set by the firmare, based upon the recording
--    criteria from within this script.
--  * It is not possible to record all samples, but all 1042 samples per second are
--    processed here in this script.
--  * To achieve high throughput, the A/D-samples are buffered in a firmware queue (buffer).
--    As a consequence, the processing of the A/D-samples is not synchronous to the state
--    of the "state machine" in this script (see "Part I" above). 
--    In effect, "Part II" of this script (Sample Processing) always lags behind Part I
--    (State Machine) by a few milliseconds. It may be regarded as a seperate thread.
--    Thus the script must be able to correctly handle A/D-samples that were recorded at the 
--    end of the previous step. To make that possible, the script checks the time stamps
--    of the samples and then assigns the corresponding step and cycle numbers to the samples
--    (see variables sample_step_tlast, sample_step_n, and sample_cycle_n).
---------------------------------------------------------------------------------
function NewSample(...)

    if started and processing and not paused then
        
        arg = {...}  -- argument array

        sample_t       = arg[1]
        V12            = arg[2]
        V1R            = arg[3]
        V2R            = arg[4]
        i              = arg[5]
        Vaux           = arg[6]
        sample_over    = arg[9]    -- string
        sample_i_range = arg[10]   -- integer
        sample_rec     = arg[11]   -- boolean flag: record this sample
        sample_queue   = arg[12]   -- integer
        
        -- call subroutines to process the sample --
        
        -- do all math calculations
        CalculateSampleValues()
        
        -- execute user calculations and code
        if stepconf[sample_step_n].doPerSample then stepconf[sample_step_n].doPerSample(sample_step_n) end
        
        -- execute limit functions (run the state machine)
        CheckLimits()

        -- check if dV is sufficient to calculate differential capacity
        DiffCap()

        -- record sample if...
        sample_rec = 
               sample_rec   -- if internal reduction algorithm has flagged the sample as "important", based on the record criteria
            or ((sample_t - sample_t_prev_record) > record_maximum_dt)  -- if delta-t has become too long (this becomes obsolete in 2023, better use record criteria with "t")
            or IsFirstSampleOfStep() 
            or IsLastSampleOfStep() 

        -- A) output sample for recording to Output.txt file AND for live display
        if sample_rec then
            sample_t_prev_record = sample_t
            sample_t_prev_display = sample_t
            PrintOutput()
        -- B) output sample for for live display only
        elseif ((sample_t - sample_t_prev_display) > display_maximum_dt) then
            sample_t_prev_display = sample_t
            PrintOutput() 
        end
        -- record sensordata --
        if (sample_t - sensor_sample_t_prev_record) > record_sensor_dt then
            sensor_sample_t_prev_record = sample_t
            PrintOutputSensordata()
        end
        
        -- advance step and/or cycle? --
        if IsLastSampleOfStep() then
            -- this was the last sample of the step
            -- check if it is also the last sample of the cycle --
            if sample_cycle_n == 0 and cycle_n == 1 then
                InitNewCycle()    -- begin processing the first cycle
            elseif (sample_cycle_n < cycle_n) or finished then  -- cycle transition
                OutputCycleData() -- print to Output-Cycles.txt 
                InitNewCycle()    -- begin processing the next cycle
            end

            if not finished then
                InitSamplingStepState()  -- init the next sample step
                ec.print(SEV_INFO, "Step "..step_n.." ("..step_type..") is processing")
            else
                -- statemachine has finished the last step
                -- (there is no next step or the next step is "OFF" and has no limits)
                processing = false  -- do not accept any more new samples
                ec.SetScriptExecutionState("Finished")  -- this has to be the very last message
            end
        end   
    end
end

----------------- IsLastSampleOfStep --------------------------------------------
-- Checks if A/D sample is the last of the step
---------------------------------------------------------------------------------
function IsLastSampleOfStep()
    return (sample_t == sample_step_tlast)
end

function IsFirstSampleOfStep()
    local x = next_sample_is_first_sample_of_step
    next_sample_is_first_sample_of_step = false
    return x
end

-------------------- Init Sampling Step State -----------------------------------
-- Updates the sample state.
-- Since the processing of the A/D-samples lags behind of the state of the
-- state machine, the correct state that belongs to the A/D-samples is set here.
---------------------------------------------------------------------------------
function InitSamplingStepState()

    -- update sample step state
    sample_step_n      = step_n
    sample_step_tstart = step_tstart    -- time of PGStat switching via ec.SwitchNow()
    sample_cycle_tstart = cycle_tstart  -- time of PGStat switching via ec.SwitchNow()
    sample_step_type   = step_type        or "-"
    sample_step_pg     = step_pg          or "-"
    sample_step_con    = step_con         or "-"
    sample_step_dV     = step_diffcap_dV  or 0.001
    sample_cds         = step_cds         or "-"
    sample_cd          = step_cd          or "-"
    sample_cycle_n     = cycle_n
    
    next_sample_is_first_sample_of_step = true  -- always keep the first sample of a new step

    -- execute user code here --
    if stepconf[sample_step_n].doAtStepStart then stepconf[sample_step_n].doAtStepStart(sample_step_n) end
end

-------------------- Calculate Sample Values ------------------------------------
-- This subroutine calculates power, charge and energy.
---------------------------------------------------------------------------------
function CalculateSampleValues()

    sample_step_t  = sample_t - sample_step_tstart
    sample_cycle_t = sample_t - sample_cycle_tstart

    local dt  = sample_t - sample_t_prev    -- delta t
    local dQ  = i * dt                      -- delta Q (charge)
    local dW  = 0                           -- delta W (energy)

    -- calculate variables that depend on the electrode connection setting "con" --
    if sample_step_con == "1vs2" or 
       sample_step_con == "1vsR" or 
       sample_step_con == "2vsR" then
       
        i12 = i
        i1R = 0
        i2R = 0
        P   = V12 * i   -- P12
        dW  = P   * dt
        Q12 = Q12 + dQ
        W12 = W12 + dW
        
    elseif sample_step_con == "Rvs1" then
        i12 = 0
        i1R = i
        i2R = 0
        P   = V1R * i   -- P1R
        dW  = P   * dt  -- for half cell energy efficiency "Eff_W" only
        Q1R = Q1R + dQ
        
    elseif sample_step_con == "Rvs2" then
        i12 = 0
        i1R = 0
        i2R = i
        P   = V2R * i   -- P2R
        dW  = P   * dt  -- for half cell energy efficiency "Eff_W" only
        Q2R = Q2R + dQ
        
    else -- OCV
        i12 = 0
        i1R = 0
        i2R = 0
        P   = 0
    end
    
    -- calculate electrode charge --    
    Q1 =  Q12 + Q1R       
    Q2 = -Q12 + Q2R    
    QR = -Q1R - Q2R     
    
    -- calculate variables that are independent of "con" --
    if sample_cd == "c" then
        --charging--
        Qc  = Qc + dQ  -- add positive charge --> Qc is always positive
        Wc  = Wc + dW  -- add positive energy --> Wc is always positive
        Qcd = Qc
    elseif sample_cd == "d" then
        --discharging--
        Qd  = Qd - dQ  -- subtract negative charge --> Qd is always positive
        Wd  = Wd - dW  -- subtract negative energy --> Wd is always positive
        Qcd = -Qd
    end
    Qcd_abs = math.abs(Qcd)
    
    
    -- calculate low-pass-filtered values --
    LowpassFilter()
    
    -- calculate |dV/dt|, if sample_rec, or once every second --
    local _dt = sample_t - (dvdt_t_prev or -math.huge)
    if (sample_rec and (_dt > delta_t_min)) or (_dt > delta_t_max) then 
        local dV12 = V12_avg - (V12_avg_prev or V12_avg)
        local dV1R = V1R_avg - (V1R_avg_prev or V1R_avg)
        local dV2R = V2R_avg - (V2R_avg_prev or V2R_avg)
        dV12_dt_abs = math.abs(dV12 / _dt) -- |dV12/dt|
        dV1R_dt_abs = math.abs(dV1R / _dt) -- |dV1R/dt|
        dV2R_dt_abs = math.abs(dV2R / _dt) -- |dV2R/dt|
        V12_avg_prev = V12_avg
        V1R_avg_prev = V1R_avg
        V2R_avg_prev = V2R_avg
        dvdt_t_prev = sample_t
    end
    
    sample_t_prev = sample_t   -- remember sample time for next delta t
end

-------------------- Lowpass Filter  --------------------------------------------
-- Averages the voltages over one second. 
-- Used for i.e. Differential Capacity.
---------------------------------------------------------------------------------
function LowpassFilter()

    -- 1000 milliseconds filter time constant
    
    if avg_counter < 1000 then avg_counter = avg_counter + 1 end
    
    if avg_counter == 1 then
        V12_avg = V12 -- start value
        V1R_avg = V1R 
        V2R_avg = V2R
    else
        V12_avg = V12_avg - (V12_avg / avg_counter)
        V1R_avg = V1R_avg - (V1R_avg / avg_counter)
        V2R_avg = V2R_avg - (V2R_avg / avg_counter)

        V12_avg = V12_avg + (V12 / avg_counter)
        V1R_avg = V1R_avg + (V1R / avg_counter)
        V2R_avg = V2R_avg + (V2R / avg_counter)
    end
end

-------------------- Differential Capacity  -------------------------------------
--  Checks if sample_step_dV is sufficient to calculate differential capacity and 
--  eventually calculates and prints differential capacity data.
---------------------------------------------------------------------------------
function DiffCap()

    if  (sample_step_dV)  -- check if variable is defined
    and (sample_step_type == "CC" or sample_step_type == "VS") then  -- diff cap makes no sense when in "CV" modes and is skipped during GEIS
    
        if sample_step_t < step_diffcap_startdelayR then 
        
            -- reset variables if diffcap_startdelayR is larger than zero
            -- and keep variables in reset for diffcap_startdelayR seconds
            if (sample_step_con == "1vs2"
             or sample_step_con == "1vsR"
             or sample_step_con == "2vsR") then  -- current is i12
                last_V_C12 = V12_avg
                last_V_C1  = V1R_avg
                last_V_C2  = V2R_avg
                last_Q_C12 = Q12
                last_Q_C1  = Q1
                last_Q_C2  = Q2
            elseif (sample_step_con == "Rvs1") then  -- current is i1R
                last_V_C1R = V1R_avg
                last_V_C1  = V12_avg
                last_V_CR  = V2R_avg
                last_Q_C1R = Q1R
                last_Q_C1  = Q1
                last_Q_CR  = QR
            elseif (sample_step_con == "Rvs2") then  -- current is i2R
                last_V_C2R = V2R_avg
                last_V_C2  = V12_avg
                last_V_CR  = V1R_avg
                last_Q_C2R = Q2R
                last_Q_C2  = Q2
                last_Q_CR  = QR
            end
            
        else
            if sample_t > (sample_t_prev_diffcap + 1)  -- call this code only once per second
            and sample_step_t > step_diffcap_startdelay then 
                DiffCap_Check_Calculate_Print()
            end
        end
        
    end
end

function DiffCap_Check_Calculate_Print()

    -- this code is called only once per second --
    sample_t_prev_diffcap = sample_t
    local dV

    if (sample_step_con == "1vs2"
     or sample_step_con == "1vsR"
     or sample_step_con == "2vsR") then  -- current i12
        -- C12 --
        dV     = math.abs(V12_avg - last_V_C12)
        dQ_C12 = Q12     - last_Q_C12
        if dV > sample_step_dV then
            C12 = dQ_C12 / dV
            PrintOutputC12()
            last_V_C12 = V12_avg
            last_Q_C12 = Q12
        end
        -- C1 --
        dV    = math.abs(V1R_avg - last_V_C1) 
        dQ_C1 = Q1      - last_Q_C1
        if dV > sample_step_dV then
            C1 = dQ_C1 / dV
            PrintOutputC1()
            last_V_C1 = V1R_avg
            last_Q_C1 = Q1
        end
        -- C2 --
        dV    = math.abs(V2R_avg - last_V_C2)
        dQ_C2 = Q2      - last_Q_C2
        if dV > sample_step_dV then
            C2 = dQ_C2 / dV
            PrintOutputC2()
            last_V_C2 = V2R_avg
            last_Q_C2 = Q2
        end
        
    elseif (sample_step_con == "Rvs1") then  -- current i1R
        -- C1R --
        dV     = math.abs(V1R_avg - last_V_C1R)
        dQ_C1R = Q1R     - last_Q_C1R
        if dV > sample_step_dV then
            C1R = dQ_C1R / dV
            PrintOutputC1R()
            last_V_C1R = V1R_avg
            last_Q_C1R = Q1R
        end
        -- C1 --
        dV    = math.abs(V12_avg - last_V_C1)
        dQ_C1 = Q1      - last_Q_C1
        if dV > sample_step_dV then
            C1 = dQ_C1 / dV
            PrintOutputC1()
            last_V_C1 = V12_avg
            last_Q_C1 = Q1
        end
        -- CR --
        dV    = math.abs(V2R_avg - last_V_CR)
        dQ_CR = QR      - last_Q_CR
        if dV > sample_step_dV then
            CR = dQ_CR / dV
            PrintOutputCR()
            last_V_CR = V2R_avg
            last_Q_CR = QR
        end
        
    elseif (sample_step_con == "Rvs2") then  -- current i2R
        -- C2R --
        dV     = math.abs(V2R_avg - last_V_C2R)
        dQ_C2R = Q2R     - last_Q_C2R
        if dV > sample_step_dV then
            C2R = dQ_C2R / dV
            PrintOutputC2R()
            last_V_C2R = V2R_avg
            last_Q_C2R = Q2R    
        end
        -- C2 --
        dV    = math.abs(V12_avg - last_V_C2)
        dQ_C2 = Q2      - last_Q_C2
        if dV > sample_step_dV then
            C2 = dQ_C2 / dV
            PrintOutputC2()
            last_V_C2 = V12_avg
            last_Q_C2 = Q2
        end
        -- CR --
        dV    = math.abs(V1R_avg - last_V_CR)
        dQ_CR = QR      - last_Q_CR
        if dV > sample_step_dV then
            CR = dQ_CR / dV
            PrintOutputCR()
            last_V_CR = V1R_avg
            last_Q_CR = QR
        end
    end
end

-------------------- Output Cycle Data ------------------------------------------
-- This subroutine is called when a complete cycle has been finished.
---------------------------------------------------------------------------------
function OutputCycleData()
    -- calculate charge efficiency --
    if sample_cds == "d" then
        -- cycles start with discharge --
        EffQ = Qc / Qd   
        EffW = Wc / Wd   
    else
        -- cycles start with charge --
        EffQ = Qd / Qc   
        EffW = Wd / Wc   
    end

    -- print cycle data to file --
    PrintOutputCycles()
    
    -- do a check on electrode charge: Q1+Q2+QR = 0 ? --
    if math.abs(Q1 + Q2 + QR) > 0.00001 then
        ec.print(SEV_ERR, "Q1+Q2+QR does not equal zero.")
    end
end

function InitNewCycle()
    
    Qc = 0  -- null charge
    Qd = 0  -- null charge
    Wc = 0  -- null energy
    Wd = 0  -- null energy
    
    if not finished then ec.print(SEV_INFO, "Cycle "..cycle_n.." is processing") end
end

----------------- New EIS Data (function called by firmware) --------------------
-- The channel-firmware calls this routine after every single EIS frequency measurement.
-- See above for variable definitions
---------------------------------------------------------------------------------
function NewEISData(...)

    imp_arg = {...}   -- argument is an array of impedance data numbers (60 arguments * 16 bytes = 960 bytes)
        
    EIS_freq     = imp_arg[1]
    EIS_t        = imp_arg[2]
    EIS_V12_dc   = imp_arg[7]
    EIS_V1R_dc   = imp_arg[8]
    EIS_V2R_dc   = imp_arg[9]
    EIS_i12_dc   = imp_arg[10]
    EIS_i1R_dc   = imp_arg[11]
    EIS_i2R_dc   = imp_arg[12]
    EIS_i_range  = imp_arg[13]
    EIS_Z12_re   = imp_arg[15]  EIS_Z12_im = imp_arg[16]  EIS_Z12_mag = imp_arg[17]  EIS_Z12_phi = imp_arg[18]  EIS_dV12 = imp_arg[19]  EIS_di12 = imp_arg[20]  EIS_Z12_SFDR = imp_arg[21]
    EIS_Z1_re    = imp_arg[23]  EIS_Z1_im  = imp_arg[24]  EIS_Z1_mag  = imp_arg[25]  EIS_Z1_phi  = imp_arg[26]  EIS_dV1R = imp_arg[27]                          EIS_Z1_SFDR  = imp_arg[28]
    EIS_Z2_re    = imp_arg[30]  EIS_Z2_im  = imp_arg[31]  EIS_Z2_mag  = imp_arg[32]  EIS_Z2_phi  = imp_arg[33]  EIS_dV2R = imp_arg[34]                          EIS_Z2_SFDR  = imp_arg[35]
    EIS_Z1R_re   = imp_arg[37]  EIS_Z1R_im = imp_arg[38]  EIS_Z1R_mag = imp_arg[39]  EIS_Z1R_phi = imp_arg[40]                          EIS_di1R = imp_arg[41]  EIS_Z1R_SFDR = imp_arg[42]
    EIS_Z2R_re   = imp_arg[44]  EIS_Z2R_im = imp_arg[45]  EIS_Z2R_mag = imp_arg[46]  EIS_Z2R_phi = imp_arg[47]                          EIS_di2R = imp_arg[48]  EIS_Z2R_SFDR = imp_arg[49]
    EIS_Cs       = imp_arg[51]
    EIS_Cp       = imp_arg[52]
    EIS_precycles= imp_arg[53]
    EIS_averages = imp_arg[54]
    EIS_DAC      = imp_arg[55] -- for debug only
    EIS_A        = imp_arg[56] -- for debug only
    EIS_n        = imp_arg[57] -- for debug only
    EIS_m        = imp_arg[58] -- for debug only
    EIS_f_err    = imp_arg[59] -- for debug only
    EIS_dt       = imp_arg[60] -- for debug only
    
    collectgarbage('collect')
    
    PrintOutputImpedance(false)
end


--#############################################################################--
--##                                                                         ##--
--##                  P A R T  3 :   L O G   F I L E S                       ##--
--##                                                                         ##--
--#############################################################################--

--------------------------- Send Output Data ------------------------------------
--  Plotted as y over x=time, with variable time period T (see reduction algorithm).
--  Sends a complete sample to EL-Software-Server.
--  If the parameter "header" is true, then this function will only print the header.
---------------------------------------------------------------------------------
function PrintOutput(header)
    local file = "Output.txt"
    local rec = sample_rec
    if header then
        -- send CSV-file-header in JSON format --
        ec.Write_CSV_Header(file, [[
        {"Name": "t"      , "Unit": "s"   , "FieldType" : "timespan"},
        {"Name": "t_cycle", "Unit": "s"   , "FieldType" : "timespan"},
        {"Name": "t_step" , "Unit": "s"   , "FieldType" : "timespan"},
        {"Name": "over"   , "Unit": ""    , "FieldType" : "text"    },
        {"Name": "cycle"  , "Unit": ""    , "FieldType" : "integer" },
        {"Name": "step"   , "Unit": ""    , "FieldType" : "integer" },
        {"Name": "type"   , "Unit": ""    , "FieldType" : "text"    },
        {"Name": "p/g"    , "Unit": ""    , "FieldType" : "text"    },
        {"Name": "c/d"    , "Unit": ""    , "FieldType" : "text"    },
        {"Name": "con"    , "Unit": ""    , "FieldType" : "text"    },
        {"Name": "I-Range", "Unit": ""    , "FieldType" : "integer" },
        {"Name": "V12"    , "Unit": "V"   , "FieldType" : "decimal" },
        {"Name": "V1R"    , "Unit": "V"   , "FieldType" : "decimal" },
        {"Name": "V2R"    , "Unit": "V"   , "FieldType" : "decimal" },
        {"Name": "I12"    , "Unit": "mA"  , "FieldType" : "decimal" },
        {"Name": "I1R"    , "Unit": "mA"  , "FieldType" : "decimal" },
        {"Name": "I2R"    , "Unit": "mA"  , "FieldType" : "decimal" },
        {"Name": "Qcd"    , "Unit": "mAh" , "FieldType" : "decimal" },
        {"Name": "|Qcd|"  , "Unit": "mAh" , "FieldType" : "decimal" },
        {"Name": "Qc"     , "Unit": "mAh" , "FieldType" : "decimal" },
        {"Name": "Qd"     , "Unit": "mAh" , "FieldType" : "decimal" },
        {"Name": "Q1"     , "Unit": "mAh" , "FieldType" : "decimal" },
        {"Name": "Q2"     , "Unit": "mAh" , "FieldType" : "decimal" },
        {"Name": "QR"     , "Unit": "mAh" , "FieldType" : "decimal" },
        {"Name": "P"      , "Unit": "mW"  , "FieldType" : "decimal" },
        {"Name": "W12"    , "Unit": "mWh" , "FieldType" : "decimal" },
        {"Name": "Vaux"   , "Unit": "V"   , "FieldType" : "decimal" },
        {"Name": "T_cell" , "Unit": "degC", "FieldType" : "decimal" },
        {"Name": "Pressure","Unit": "mbar", "FieldType" : "decimal" },
        {"Name": "Dilation","Unit": "um"  , "FieldType" : "decimal" },
        {"Name": "Force"  , "Unit": "N"   , "FieldType" : "decimal" },
        {"Name": "buf"    , "Unit": ""    , "FieldType" : "integer" }]])
    end

    tim_data[1]  = sample_t
    tim_data[2]  = sample_cycle_t
    tim_data[3]  = sample_step_t
    tim_data[4]  = sample_over
    tim_data[5]  = sample_cycle_n
    tim_data[6]  = sample_step_n
    tim_data[7]  = sample_step_type
    tim_data[8]  = sample_step_pg
    tim_data[9]  = sample_cd
    tim_data[10] = sample_step_con
    tim_data[11] = sample_i_range
    tim_data[12] = V12
    tim_data[13] = V1R
    tim_data[14] = V2R
    tim_data[15] = i12 * 1000
    tim_data[16] = i1R * 1000
    tim_data[17] = i2R * 1000
    tim_data[18] = Qcd / 3.6
    tim_data[19] = Qcd_abs / 3.6
    tim_data[20] = Qc  / 3.6
    tim_data[21] = Qd  / 3.6
    tim_data[22] = Q1  / 3.6
    tim_data[23] = Q2  / 3.6
    tim_data[24] = QR  / 3.6
    tim_data[25] = P   * 1000
    tim_data[26] = W12 / 3.6
    tim_data[27] = Vaux
    tim_data[28] = T_cell
    tim_data[29] = Pressure * 1000
    tim_data[30] = Dilation * 1000000
    tim_data[31] = Force
    tim_data[32] = sample_queue

    tim_data_row = table.concat(tim_data, "\t")
    if not header then
        ec.write(file, rec, tim_data_row)
    end
end

----------------------------- Send Sensor Data ----------------------------------
--  Plotted as y over x=time, but with constant time period T = 1s.
--  This is send once every second, independent from the reduction algorithm.
--  If the parameter "header" is true, then this function will only print the header.
---------------------------------------------------------------------------------
function PrintOutputSensordata(header)
    local file = "Output-Sensordata.txt"
    local rec = true
    if header then
        ec.Write_CSV_Header(file, [[
        {"Name": "t"      , "Unit": "s"   , "FieldType" : "timespan"},
        {"Name": "t_cycle", "Unit": "s"   , "FieldType" : "timespan"},
        {"Name": "t_step" , "Unit": "s"   , "FieldType" : "timespan"},
        {"Name": "T_set"    , "Unit": "degC", "FieldType" : "decimal" },
        {"Name": "T_chamber", "Unit": "degC", "FieldType" : "decimal" },
        {"Name": "T_cell"   , "Unit": "degC", "FieldType" : "decimal" },
        {"Name": "Pressure" , "Unit": "mbar", "FieldType" : "decimal" },
        {"Name": "Dilation" , "Unit": "um"  , "FieldType" : "decimal" },
        {"Name": "Force"    , "Unit": "N"   , "FieldType" : "decimal" },
        {"Name": "T_board"  , "Unit": "degC", "FieldType" : "decimal" },
        {"Name": "T_ambient", "Unit": "degC", "FieldType" : "decimal" }]])
    end
    sen_data[1]  = sample_t
    sen_data[2]  = sample_cycle_t
    sen_data[3]  = sample_step_t
    sen_data[4]  = T_set
    sen_data[5]  = T_chamber
    sen_data[6]  = T_cell
    sen_data[7]  = Pressure * 1000
    sen_data[8]  = Dilation * 1000000
    sen_data[9]  = Force
    sen_data[10] = T_board
    sen_data[11] = T_ambient
    
    sensor_data_row = table.concat(sen_data, "\t")
    if not header then
        ec.write(file, rec, sensor_data_row)
    end
end

-------------------- Send Differential Capacity Data ----------------------------
--  Plotted as y = C12 over x = V12
--  If the parameter "header" is true, then this function will only print the header.
---------------------------------------------------------------------------------
function PrintOutputC12(header)
    local file = "Output-C12.txt"
    local rec = true
    if header then
        ec.Write_CSV_Header(file, [[
        {"Name": "t"          , "Unit": "s"    , "FieldType" : "timespan"},
        {"Name": "cycle"      , "Unit": ""     , "FieldType" : "integer" },
        {"Name": "step"       , "Unit": ""     , "FieldType" : "integer" },
        {"Name": "V12"        , "Unit": "V"    , "FieldType" : "decimal" },
        {"Name": "dQ12"       , "Unit": "uAh"  , "FieldType" : "decimal" },
        {"Name": "C12"        , "Unit": "mAh/V", "FieldType" : "decimal" }]])
    end
    C12_data[1] = sample_t
    C12_data[2] = sample_cycle_n
    C12_data[3] = sample_step_n
    C12_data[4] = V12_avg
    C12_data[5] = dQ_C12 / 0.0036
    C12_data[6] = C12    / 3.6

    C12_data_row = table.concat(C12_data, "\t")
    if not header then
        ec.write(file, rec, C12_data_row)
    end
end
function PrintOutputC1R(header)
    local file = "Output-C1R.txt"
    local rec = true
    if header then
        ec.Write_CSV_Header(file, [[
        {"Name": "t"          , "Unit": "s"    , "FieldType" : "timespan"},
        {"Name": "cycle"      , "Unit": ""     , "FieldType" : "integer" },
        {"Name": "step"       , "Unit": ""     , "FieldType" : "integer" },
        {"Name": "V1R"        , "Unit": "V"    , "FieldType" : "decimal" },
        {"Name": "dQ1R"       , "Unit": "uAh"  , "FieldType" : "decimal" },
        {"Name": "C1R"        , "Unit": "mAh/V", "FieldType" : "decimal" }]])
    end
    C1R_data[1] = sample_t
    C1R_data[2] = sample_cycle_n
    C1R_data[3] = sample_step_n
    C1R_data[4] = V1R_avg
    C1R_data[5] = dQ_C1R / 0.0036
    C1R_data[6] = C1R     / 3.6

    C1R_data_row = table.concat(C1R_data, "\t")
    if not header then
        ec.write(file, rec, C1R_data_row)
    end
    
end
function PrintOutputC2R(header)
    local file = "Output-C2R.txt"
    local rec = true
    if header then
        ec.Write_CSV_Header(file, [[
        {"Name": "t"          , "Unit": "s"    , "FieldType" : "timespan"},
        {"Name": "cycle"      , "Unit": ""     , "FieldType" : "integer" },
        {"Name": "step"       , "Unit": ""     , "FieldType" : "integer" },
        {"Name": "V2R"        , "Unit": "V"    , "FieldType" : "decimal" },
        {"Name": "dQ2R"       , "Unit": "uAh"  , "FieldType" : "decimal" },
        {"Name": "C2R"        , "Unit": "mAh/V", "FieldType" : "decimal" }]])
    end
    C2R_data[1] = sample_t
    C2R_data[2] = sample_cycle_n
    C2R_data[3] = sample_step_n
    C2R_data[4] = V2R_avg
    C2R_data[5] = dQ_C2R / 0.0036
    C2R_data[6] = C2R    / 3.6
            
    C2R_data_row = table.concat(C2R_data, "\t")
    if not header then
        ec.write(file, rec, C2R_data_row)
    end
end
function PrintOutputC1(header)
    local file = "Output-C1.txt"
    local rec = true
    if header then
        ec.Write_CSV_Header(file, [[
        {"Name": "t"          , "Unit": "s"    , "FieldType" : "timespan"},
        {"Name": "cycle"      , "Unit": ""     , "FieldType" : "integer" },
        {"Name": "step"       , "Unit": ""     , "FieldType" : "integer" },
        {"Name": "V12"        , "Unit": "V"    , "FieldType" : "decimal" },
        {"Name": "V1R"        , "Unit": "V"    , "FieldType" : "decimal" },
        {"Name": "dQ1"        , "Unit": "uAh"  , "FieldType" : "decimal" },
        {"Name": "C1"         , "Unit": "mAh/V", "FieldType" : "decimal" }]])
    end
    C1_data[1] = sample_t
    C1_data[2] = sample_cycle_n
    C1_data[3] = sample_step_n
    C1_data[4] = V12_avg
    C1_data[5] = V1R_avg
    C1_data[6] = dQ_C1 / 0.0036
    C1_data[7] = C1    / 3.6

    C1_data_row = table.concat(C1_data, "\t")
    if not header then
        ec.write(file, rec, C1_data_row)
    end
    
end
function PrintOutputC2(header)
    local file = "Output-C2.txt"
    local rec = true
    if header then
        ec.Write_CSV_Header(file, [[
        {"Name": "t"          , "Unit": "s"    , "FieldType" : "timespan"},
        {"Name": "cycle"      , "Unit": ""     , "FieldType" : "integer" },
        {"Name": "step"       , "Unit": ""     , "FieldType" : "integer" },
        {"Name": "V12"        , "Unit": "V"    , "FieldType" : "decimal" },
        {"Name": "V2R"        , "Unit": "V"    , "FieldType" : "decimal" },
        {"Name": "dQ2"        , "Unit": "uAh"  , "FieldType" : "decimal" },
        {"Name": "C2"         , "Unit": "mAh/V", "FieldType" : "decimal" }]])
    end
    C2_data[1] = sample_t
    C2_data[2] = sample_cycle_n
    C2_data[3] = sample_step_n
    C2_data[4] = V12_avg
    C2_data[5] = V2R_avg
    C2_data[6] = dQ_C2 / 0.0036
    C2_data[7] = C2    / 3.6
            
    C2_data_row = table.concat(C2_data, "\t")
    if not header then
        ec.write(file, rec, C2_data_row)
    end
end
function PrintOutputCR(header)
    local file = "Output-CR.txt"
    local rec = true
    if header then
        ec.Write_CSV_Header(file, [[
        {"Name": "t"          , "Unit": "s"    , "FieldType" : "timespan"},
        {"Name": "cycle"      , "Unit": ""     , "FieldType" : "integer" },
        {"Name": "step"       , "Unit": ""     , "FieldType" : "integer" },
        {"Name": "V1R"        , "Unit": "V"    , "FieldType" : "decimal" },
        {"Name": "V2R"        , "Unit": "V"    , "FieldType" : "decimal" },
        {"Name": "dQR"        , "Unit": "uAh"  , "FieldType" : "decimal" },
        {"Name": "CR"         , "Unit": "mAh/V", "FieldType" : "decimal" }]])
    end
    CR_data[1] = sample_t
    CR_data[2] = sample_cycle_n
    CR_data[3] = sample_step_n
    CR_data[4] = V1R_avg
    CR_data[5] = V2R_avg
    CR_data[6] = dQ_CR / 0.0036
    CR_data[7] = CR    / 3.6
            
    CR_data_row = table.concat(CR_data, "\t")
    if not header then
        ec.write(file, rec, CR_data_row)
    end
end

-------------------------------- Send Cycle Data --------------------------------
--  Plotted as y over x = cycle number
--  If the parameter "header" is true, then this function will only print the header.
---------------------------------------------------------------------------------
function PrintOutputCycles(header)
    local file = "Output-Cycles.txt"
    local rec = true
    if header then
        ec.Write_CSV_Header(file, [[
        {"Name": "t"         , "Unit": "s"   , "FieldType" : "timespan"},
        {"Name": "t_start"   , "Unit": "s"   , "FieldType" : "timespan"},
        {"Name": "t_duration", "Unit": "s"   , "FieldType" : "timespan"},
        {"Name": "cycle"     , "Unit": ""    , "FieldType" : "integer" },
        {"Name": "Qc_cyc"    , "Unit": "mAh" , "FieldType" : "decimal" },
        {"Name": "Qd_cyc"    , "Unit": "mAh" , "FieldType" : "decimal" },
        {"Name": "Wc_cyc"    , "Unit": "mWh" , "FieldType" : "decimal" },
        {"Name": "Wd_cyc"    , "Unit": "mWh" , "FieldType" : "decimal" },
        {"Name": "Eff_Q"     , "Unit": ""    , "FieldType" : "decimal" },
        {"Name": "Eff_W"     , "Unit": ""    , "FieldType" : "decimal" }]])
    end
    cyc_data[1] = sample_t
    cyc_data[2] = sample_cycle_tstart
    cyc_data[3] = sample_cycle_t
    cyc_data[4] = sample_cycle_n
    cyc_data[5] = Qc / 3.6
    cyc_data[6] = Qd / 3.6
    cyc_data[7] = Wc / 3.6
    cyc_data[8] = Wd / 3.6
    cyc_data[9] = EffQ
    cyc_data[10]= EffW

    cycle_data_row = table.concat(cyc_data, "\t")
    if not header then
        ec.write(file, rec, cycle_data_row)
    end
end

---------------------------- Send Impedance Data --------------------------------
--  Plotted as y over x=time or as Bode plot or as Nyquist plot.
--  If the parameter "header" is true, then this function will only print the header.
---------------------------------------------------------------------------------
function PrintOutputImpedance(header)
    local file = "Output-Impedance.txt"
    local rec = true
    if header then
        ec.Write_CSV_Header(file, [[
        {"Name": "freq"       , "Unit": "Hz"  , "FieldType" : "decimal" },
        {"Name": "t"          , "Unit": "s"   , "FieldType" : "timespan"},
        {"Name": "t_cycle"    , "Unit": "s"   , "FieldType" : "timespan"},
        {"Name": "cycle"      , "Unit": ""    , "FieldType" : "integer" },
        {"Name": "step"       , "Unit": ""    , "FieldType" : "integer" },
        {"Name": "type"       , "Unit": ""    , "FieldType" : "text"    },
        {"Name": "p/g"        , "Unit": ""    , "FieldType" : "text"    },
        {"Name": "c/d"        , "Unit": ""    , "FieldType" : "text"    },
        {"Name": "con"        , "Unit": ""    , "FieldType" : "text"    },
        {"Name": "I-Range"    , "Unit": ""    , "FieldType" : "integer" },
        {"Name": "V12"        , "Unit": "V"   , "FieldType" : "decimal" },
        {"Name": "V1R"        , "Unit": "V"   , "FieldType" : "decimal" },
        {"Name": "V2R"        , "Unit": "V"   , "FieldType" : "decimal" },
        {"Name": "I12"        , "Unit": "mA"  , "FieldType" : "decimal" },
        {"Name": "I1R"        , "Unit": "mA"  , "FieldType" : "decimal" },
        {"Name": "I2R"        , "Unit": "mA"  , "FieldType" : "decimal" },
        {"Name": "Q1"         , "Unit": "mAh" , "FieldType" : "decimal" },
        {"Name": "Q2"         , "Unit": "mAh" , "FieldType" : "decimal" },
        {"Name": "QR"         , "Unit": "mAh" , "FieldType" : "decimal" },
        {"Name": "Qcd"        , "Unit": "mAh" , "FieldType" : "decimal" },
        {"Name": "|Qcd|"      , "Unit": "mAh" , "FieldType" : "decimal" },
        {"Name": "Qc"         , "Unit": "mAh" , "FieldType" : "decimal" },
        {"Name": "Qd"         , "Unit": "mAh" , "FieldType" : "decimal" },
        {"Name": "Re(Z12)"    , "Unit": "Ohm" , "FieldType" : "decimal" },
        {"Name": "Im(Z12)"    , "Unit": "Ohm" , "FieldType" : "decimal" },
        {"Name": "|Z12|"      , "Unit": "Ohm" , "FieldType" : "decimal" },
        {"Name": "phi(Z12)"   , "Unit": "deg" , "FieldType" : "decimal" },
        {"Name": "dV12"       , "Unit": "mV"  , "FieldType" : "decimal" },
        {"Name": "dI12"       , "Unit": "mA"  , "FieldType" : "decimal" },
        {"Name": "SFDR(Z12)"  , "Unit": "dB"  , "FieldType" : "decimal" },
        {"Name": "Re(Z1)"     , "Unit": "Ohm" , "FieldType" : "decimal" },
        {"Name": "Im(Z1)"     , "Unit": "Ohm" , "FieldType" : "decimal" },
        {"Name": "|Z1|"       , "Unit": "Ohm" , "FieldType" : "decimal" },
        {"Name": "phi(Z1)"    , "Unit": "deg" , "FieldType" : "decimal" },
        {"Name": "dV1R"       , "Unit": "mV"  , "FieldType" : "decimal" },
        {"Name": "SFDR(Z1)"   , "Unit": "dB"  , "FieldType" : "decimal" },
        {"Name": "Re(Z2)"     , "Unit": "Ohm" , "FieldType" : "decimal" },
        {"Name": "Im(Z2)"     , "Unit": "Ohm" , "FieldType" : "decimal" },
        {"Name": "|Z2|"       , "Unit": "Ohm" , "FieldType" : "decimal" },
        {"Name": "phi(Z2)"    , "Unit": "deg" , "FieldType" : "decimal" },
        {"Name": "dV2R"       , "Unit": "mV"  , "FieldType" : "decimal" },
        {"Name": "SFDR(Z2)"   , "Unit": "dB"  , "FieldType" : "decimal" },
        {"Name": "Re(Z1R)"    , "Unit": "Ohm" , "FieldType" : "decimal" },
        {"Name": "Im(Z1R)"    , "Unit": "Ohm" , "FieldType" : "decimal" },
        {"Name": "|Z1R|"      , "Unit": "Ohm" , "FieldType" : "decimal" },
        {"Name": "phi(Z1R)"   , "Unit": "deg" , "FieldType" : "decimal" },
        {"Name": "dI1R"       , "Unit": "mA"  , "FieldType" : "decimal" },
        {"Name": "SFDR(Z1R)"  , "Unit": "dB"  , "FieldType" : "decimal" },
        {"Name": "Re(Z2R)"    , "Unit": "Ohm" , "FieldType" : "decimal" },
        {"Name": "Im(Z2R)"    , "Unit": "Ohm" , "FieldType" : "decimal" },
        {"Name": "|Z2R|"      , "Unit": "Ohm" , "FieldType" : "decimal" },
        {"Name": "phi(Z2R)"   , "Unit": "deg" , "FieldType" : "decimal" },
        {"Name": "dI2R"       , "Unit": "mA"  , "FieldType" : "decimal" },
        {"Name": "SFDR(Z2R)"  , "Unit": "dB"  , "FieldType" : "decimal" },
        {"Name": "Cs"         , "Unit": "uF"  , "FieldType" : "decimal" },
        {"Name": "Cp"         , "Unit": "uF"  , "FieldType" : "decimal" },
        {"Name": "A"          , "Unit": ""    , "FieldType" : "decimal" },
        {"Name": "n"          , "Unit": ""    , "FieldType" : "integer" },
        {"Name": "m"          , "Unit": ""    , "FieldType" : "integer" },
        {"Name": "f_err"      , "Unit": "%"   , "FieldType" : "decimal" },
        {"Name": "dt"         , "Unit": "s"   , "FieldType" : "decimal" },
        {"Name": "pre"        , "Unit": ""    , "FieldType" : "integer" },
        {"Name": "avg"        , "Unit": ""    , "FieldType" : "integer" },
        {"Name": "DA"         , "Unit": ""    , "FieldType" : "integer" }]])
    end
local t1 = ec.GetSeconds()

    imp_data[1]  = EIS_freq
    imp_data[2]  = EIS_t
    imp_data[3]  = EIS_t - cycle_tstart
    imp_data[4]  = sample_cycle_n
    imp_data[5]  = sample_step_n
    imp_data[6]  = sample_step_type
    imp_data[7]  = sample_step_pg
    imp_data[8]  = sample_cd
    imp_data[9]  = sample_step_con
    imp_data[10] = EIS_i_range
    imp_data[11] = EIS_V12_dc
    imp_data[12] = EIS_V1R_dc
    imp_data[13] = EIS_V2R_dc
    imp_data[14] = EIS_i12_dc * 1000
    imp_data[15] = EIS_i1R_dc * 1000
    imp_data[16] = EIS_i2R_dc * 1000
    imp_data[17] = Q1      / 3.6
    imp_data[18] = Q2      / 3.6
    imp_data[19] = QR      / 3.6
    imp_data[20] = Qcd     / 3.6
    imp_data[21] = Qcd_abs / 3.6
    imp_data[22] = Qc      / 3.6
    imp_data[23] = Qd      / 3.6
    imp_data[24] = EIS_Z12_re
    imp_data[25] = EIS_Z12_im
    imp_data[26] = EIS_Z12_mag
    imp_data[27] = EIS_Z12_phi
    imp_data[28] = EIS_dV12 * 1000
    imp_data[29] = EIS_di12 * 1000
    imp_data[30] = EIS_Z12_SFDR
    imp_data[31] = EIS_Z1_re
    imp_data[32] = EIS_Z1_im
    imp_data[33] = EIS_Z1_mag
    imp_data[34] = EIS_Z1_phi
    imp_data[35] = EIS_dV1R * 1000
    imp_data[36] = EIS_Z1_SFDR
    imp_data[37] = EIS_Z2_re
    imp_data[38] = EIS_Z2_im
    imp_data[39] = EIS_Z2_mag
    imp_data[40] = EIS_Z2_phi
    imp_data[41] = EIS_dV2R * 1000
    imp_data[42] = EIS_Z2_SFDR
    imp_data[43] = EIS_Z1R_re
    imp_data[44] = EIS_Z1R_im
    imp_data[45] = EIS_Z1R_mag
    imp_data[46] = EIS_Z1R_phi
    imp_data[47] = EIS_di1R * 1000
    imp_data[48] = EIS_Z1R_SFDR
    imp_data[49] = EIS_Z2R_re
    imp_data[50] = EIS_Z2R_im
    imp_data[51] = EIS_Z2R_mag
    imp_data[52] = EIS_Z2R_phi
    imp_data[53] = EIS_di2R * 1000    
    imp_data[54] = EIS_Z2R_SFDR
    imp_data[55] = EIS_Cs * 1000000  -- uF
    imp_data[56] = EIS_Cp * 1000000  -- uF
    imp_data[57] = EIS_A  * 1000
    imp_data[58] = EIS_n
    imp_data[59] = EIS_m
    imp_data[60] = EIS_f_err
    imp_data[61] = EIS_dt
    imp_data[62] = EIS_precycles
    imp_data[63] = EIS_averages
    imp_data[64] = EIS_DAC

    impedance_data_row = table.concat(imp_data, "\t")
    if not header then
        ec.write(file, rec, impedance_data_row)
    end
end


-- Now that all functions are defined --
if not started then
    -- init:
    ec.SetScriptExecutionState("Init")
    Init_StateMachine()
    Init_SampleProcessing()  -- must be called before stepconf, to init variables used in stepconf
    Generate_StepConfiguration()
    -- start:
    StartMeasurement()
    started = true
    processing = true
    ec.SetScriptExecutionState("Running")
else
    -- update only:
    Generate_StepConfiguration()
end

-- newline
